// Load FOSSEE Optimization Toolbox
clc;
clear;
disp("Advanced Optimization with Scilab - FOSSEE Toolbox");

// Define Rastrigin function
function f = rastrigin(x)
    A = 10;
    n = length(x);
    f = A * n + sum(x.^2 - A * cos(2 * %pi * x)); // Formula (i)
endfunction

// Define the Cost Function for GA
function f = cost_function(params)
    f = rastrigin(params);
endfunction

// Define Genetic Algorithm
function [best_solution, fitness_history] = genetic_algorithm(cost_function, pop_size, generations, lb, ub)
    population = repmat(lb, pop_size, 1) + (repmat(ub, pop_size, 1) - repmat(lb, pop_size, 1)) .* rand(pop_size, length(lb)); // Formula (ii)
    fitness_history = zeros(generations, 1);  

    no_improvement_count = 0;   // Early stopping counter
    stopping_threshold = 1e-6;  // Threshold for improvement
    max_no_improvement = 10;    // Number of generations without improvement before stopping

    for gen = 1:generations
        fitness = zeros(pop_size, 1);
        for i = 1:pop_size
            fitness(i) = cost_function(population(i, :)); // Function (iii)
        end
        fitness_history(gen) = min(fitness);

        // Check for early stopping
        if gen > 1 && abs(fitness_history(gen) - fitness_history(gen-1)) < stopping_threshold
            no_improvement_count = no_improvement_count + 1;
        else
            no_improvement_count = 0;
        end
        
        if no_improvement_count >= max_no_improvement
            disp("Early stopping: No significant improvement for 10 generations.");
            break;
        end

        // Selection and Crossover
        [sorted_fitness, sorted_idx] = gsort(fitness, "g", "i");
        num_selected = max(2, floor(pop_size/2)); 
        best_individuals = population(sorted_idx(1:num_selected), :);

        offspring = best_individuals;
        num_pairs = floor(size(best_individuals, 1) / 2) * 2;
        for i = 1:2:num_pairs-1
            alpha = rand();
            offspring(i, :) = alpha * best_individuals(i, :) + (1 - alpha) * best_individuals(i+1, :);  // Formula (iv)
            offspring(i+1, :) = alpha * best_individuals(i+1, :) + (1 - alpha) * best_individuals(i, :);
        end

        // Mutation
        mutation_idx = ceil(rand() * size(offspring, 1));
        mutation_idx = min(max(mutation_idx, 1), size(offspring, 1));
        offspring(mutation_idx, :) = lb + (ub - lb) .* rand(1, length(lb)); // Function (v)

        population = [best_individuals; offspring];

        // Keep population size constant
        if size(population, 1) > pop_size then
            population = population(1:pop_size, :);
        end
    end

    // Final fitness
    final_fitness = zeros(pop_size, 1);
    for i = 1:pop_size
        final_fitness(i) = cost_function(population(i, :));
    end
    [min_val, min_idx] = min(final_fitness); // Formula (vi) 
    best_solution = population(min_idx, :);
endfunction

// Run Genetic Algorithm for Hyperparameter Optimization
pop_size = 30;   // Reduced population size for faster computation
generations = 100;
lb = [-5.12, -5.12]; // Lower bounds for Rastrigin function
ub = [5.12, 5.12];   // Upper bounds

rand("seed", 1234);
[best_params, fitness_history] = genetic_algorithm(cost_function, pop_size, generations, lb, ub);

// -------------------------------
// 📈 GRAPH 1: Fitness over Generations
// -------------------------------
clf;
scf(0);
plot(1:generations, fitness_history, "r-");
xlabel("Generations");
ylabel("Best Fitness Value");
title("GA Fitness Progress");

// -------------------------------
// 📊 GRAPH 2: 3D Surface Plot with GA Best Point
// -------------------------------
x = -5.12:0.2:5.12;
y = -5.12:0.2:5.12;
[X, Y] = ndgrid(x, y);
Z = 10 * 2 + (X.^2 - 10 * cos(2 * %pi * X)) + (Y.^2 - 10 * cos(2 * %pi * Y));

scf(1);
clf;
surf(x, y, Z);
xlabel("X"); ylabel("Y"); zlabel("Rastrigin Value");
title("Rastrigin Function Landscape");

// Plot best point on surface
param3d([best_params(1)], [best_params(2)], [rastrigin(best_params)], flag=[0,0]);

// Set 3D view angles
gca().rotation_angles = [45, 60];

// -------------------------------
// 🔴 GRAPH 3: 2D Contour Plot with GA Best Point
// -------------------------------
scf(2);
clf;
contour2d(x, y, Z, 30); // 30 contour levels
xlabel("X"); ylabel("Y");
title("GA Solution on Rastrigin Contour");
xgrid(1);

// Plot the best point (cross marker)
xpoly([best_params(1)], [best_params(2)], "marks", 3);

// Display Best Hyperparameters and Fitness Value
disp("Best Hyperparameters Found:");
disp(best_params);

final_value = rastrigin(best_params);
disp("Best Objective Function Value (Rastrigin):");
disp(final_value);


