// Function Definitions 

// Gauss-Seidel solver for an external mesh
function [phi, res_sum] = FVM_GS_ext_mesh(Nx, Ny, alpha, Nit_max, tol, phi)
    global aW aE aS aN aP bP;
    res_sum = 1e+6;  // initialize residual
    for Nit = 1:Nit_max
        // Update interior nodes
        for J = 2:Ny
            for I = 2:Nx
                phi(I,J) = ( aW(I,J)*phi(I-1,J) + aE(I,J)*phi(I+1,J) + ...
                             aS(I,J)*phi(I,J-1) + aN(I,J)*phi(I,J+1) + bP(I,J) )/(aP(I,J)/alpha) ...
                             + (1 - alpha)*phi(I,J);
            end
        end
        // Compute residuals
        res = zeros(Nx-1, Ny-1);
        for J = 2:Ny
            for I = 2:Nx
                res(I-1, J-1) = abs( aW(I,J)*phi(I-1,J) + aE(I,J)*phi(I+1,J) + ...
                                     aS(I,J)*phi(I,J-1) + aN(I,J)*phi(I,J+1) + bP(I,J) - aP(I,J)*phi(I,J) );
            end
        end
        // Normalize residual (avoid division by zero)
        denom = sum(sum(abs(aP(2:Nx,2:Ny) .* phi(2:Nx,2:Ny))));
        if denom == 0 then
            denom = 1;
        end
        res_sum = sum(sum(res)) / denom;
        
        if res_sum < tol then
            // Convergence achieved, exit the loop
            break;
        end
    end
endfunction

// Pressure-correction equation setup
function FVM_pcorr(Nx, Ny, dx, dy, dz, rho, Ip, Jp, uStar, vStar)
    global aE aW aN aS aP bP dU dV;
    aE(Ip,Jp) = rho * dU(Ip, Jp) * dy * dz;
    aW(Ip,Jp) = rho * dU(Ip-1, Jp) * dy * dz;
    aN(Ip,Jp) = rho * dV(Ip, Jp) * dx * dz;
    aS(Ip,Jp) = rho * dV(Ip, Jp-1) * dx * dz;
    aP(Ip,Jp) = aE(Ip,Jp) + aW(Ip,Jp) + aN(Ip,Jp) + aS(Ip,Jp);
    bP(Ip,Jp) = rho*(uStar(Ip-1,Jp) - uStar(Ip,Jp))*dy*dz + ...
                rho*(vStar(Ip,Jp-1) - vStar(Ip,Jp))*dx*dz;
    // Fix pressure at outlet to zero:
    i = Nx+1;
    aE(i,Jp) = 0; aW(i,Jp) = 0; aN(i,Jp) = 0; aS(i,Jp) = 0;
    aP(i,Jp) = 1; bP(i,Jp) = 0;
endfunction

// Temperature (phi) equation coefficients and assembly
function FVM_phi(Nx, Ny, dx, dy, dz, rho, gamma, dphi, Ip, Jp, u, v, BC_S, BC_N)
    global Fw Fe Fn Fs DF aE aW aN aS aP bP;
    Dx = (gamma/dx)*(dy*dz);
    Dy = (gamma/dy)*(dx*dz);
    
    Fe(Ip,Jp) = rho*dy*dz*u(Ip,Jp);
    Fw(Ip,Jp) = rho*dy*dz*u(Ip-1,Jp);
    Fn(Ip,Jp) = rho*dx*dz*v(Ip,Jp);
    Fs(Ip,Jp) = rho*dx*dz*v(Ip,Jp-1);
    
    // Hybrid upwinding for convection terms
    for i = Ip
        for j = Jp
            aE(i,j) = max([ -Fe(i,j), (Dx - 0.5*Fe(i,j)), 0 ]);
            aW(i,j) = max([  Fw(i,j), (Dx + 0.5*Fw(i,j)), 0 ]);
            aN(i,j) = max([ -Fn(i,j), (Dy - 0.5*Fn(i,j)), 0 ]);
            aS(i,j) = max([  Fs(i,j), (Dy + 0.5*Fs(i,j)), 0 ]);
        end
    end
    bP(Ip,Jp) = 0;
    if BC_N == 0 then
        aN(:,Ny+1) = 2*Dy;
    else
        aN(:,Ny+1) = 0;
        bP(:,Ny+1) = dphi*dx*dz;
    end
    if BC_S == 0 then
        aS(:,2) = 2*Dy;
    else
        aS(:,2) = 0;
    end
    aE(Nx+1,:) = 0; // outlet boundary (dT/dx=0)
    DF(Ip,Jp) = Fe(Ip,Jp) - Fw(Ip,Jp) + Fn(Ip,Jp) - Fs(Ip,Jp);
    aP(Ip,Jp) = aE(Ip,Jp) + aW(Ip,Jp) + aN(Ip,Jp) + aS(Ip,Jp) + DF(Ip,Jp);
endfunction

// x-momentum (u-velocity) equation setup
function u_out = FVM_u(Nx, Ny, dx, dy, dz, rho, Dx, Dy, iF, Ju, alphaU, u, v, pStar, BC_S_MO)
    global Fw Fe Fn Fs DF aE aW aN aS aP bP dU;
    // Compute face fluxes for convection
    Fw(iF,Ju) = rho*dy*dz*0.5*(u(iF,Ju) + u(iF-1,Ju));
    Fe(iF,Ju) = rho*dy*dz*0.5*(u(iF,Ju) + u(iF+1,Ju));
    Fs(iF,Ju) = rho*dx*dz*0.5*(v(iF,Ju-1) + v(iF+1,Ju-1));
    Fn(iF,Ju) = rho*dx*dz*0.5*(v(iF,Ju) + v(iF+1,Ju));
    for i = iF
        for j = Ju
            aW(i,j) = max([ Fw(i,j), (Dx + 0.5*Fw(i,j)), 0 ]);
            aE(i,j) = max([ -Fe(i,j), (Dx - 0.5*Fe(i,j)), 0 ]);
            aS(i,j) = max([ Fs(i,j), (Dy + 0.5*Fs(i,j)), 0 ]);
            aN(i,j) = max([ -Fn(i,j), (Dy - 0.5*Fn(i,j)), 0 ]);
        end
    end
    if BC_S_MO == 0 then
        aS(:,2) = 2*Dy;
    else
        aS(:,2) = 0;
    end
    aN(:,Ny+1) = 2*Dy;   // top wall condition
    aE(Nx+1,:) = 0;      // outlet: du/dx = 0
    DF(iF,Ju) = Fe(iF,Ju) - Fw(iF,Ju) + Fn(iF,Ju) - Fs(iF,Ju);
    aP(iF,Ju) = aE(iF,Ju) + aW(iF,Ju) + aN(iF,Ju) + aS(iF,Ju) + DF(iF,Ju);
    bP(iF,Ju) = dy*dz*(pStar(iF,Ju) - pStar(iF+1,Ju));
    dU(iF,Ju) = (dy*dz) ./ (aP(iF,Ju)/alphaU);
    u_out = u;
endfunction

// y-momentum (v-velocity) equation setup
function v_out = FVM_v(Nx, Ny, dx, dy, dz, rho, Dx, Dy, Iv, jF, alphaU, u, v, pStar)
    global Fw Fe Fn Fs DF aE aW aN aS aP bP dV;
    Fw(Iv,jF) = rho*dy*dz*0.5*(u(Iv-1,jF) + u(Iv-1,jF+1));
    Fe(Iv,jF) = rho*dy*dz*0.5*(u(Iv,jF) + u(Iv,jF+1));
    Fs(Iv,jF) = rho*dx*dz*0.5*(v(Iv,jF) + v(Iv,jF-1));
    Fn(Iv,jF) = rho*dx*dz*0.5*(v(Iv,jF) + v(Iv,jF+1));
    for i = Iv
        for j = jF
            aW(i,j) = max([ Fw(i,j), (Dx + 0.5*Fw(i,j)), 0 ]);
            aE(i,j) = max([ -Fe(i,j), (Dx - 0.5*Fe(i,j)), 0 ]);
            aS(i,j) = max([ Fs(i,j), (Dy + 0.5*Fs(i,j)), 0 ]);
            aN(i,j) = max([ -Fn(i,j), (Dy - 0.5*Fn(i,j)), 0 ]);
        end
    end
    aE(Nx+1,:) = 0;    // outlet: dv/dx = 0
    DF(Iv,jF) = Fe(Iv,jF) - Fw(Iv,jF) + Fn(Iv,jF) - Fs(Iv,jF);
    aP(Iv,jF) = aE(Iv,jF) + aW(Iv,jF) + aN(Iv,jF) + aS(Iv,jF) + DF(Iv,jF);
    bP(Iv,jF) = dx*dz*(pStar(Iv,jF) - pStar(Iv,jF+1));
    dV(Iv,jF) = (dx*dz) ./ (aP(Iv,jF)/alphaU);
    v_out = v;
endfunction
