funcprot(0)
// ---------------------------------------------
// 1. Hard-Coded Image Loading Function
// ---------------------------------------------
function img = loadImageHardCoded()
// Hard-coded file path (use forward slashes)
// Prompt user to choose an image file
fullPath = uigetfile(["*.png";"*.bmp";"*.jpg"], "Select a brain scan image");

    // Read the image using imread (ensure imread is available)
    img = imread(fullPath);
    // Convert to double precision and normalize to [0,1]
    img = double(img) / 255;
endfunction

// ---------------------------------------------
// 2. RGB to Grayscale Conversion
// ---------------------------------------------
function gray = rgbToGray(img)
    if ndims(img) == 3 then
        // Use standard weights for conversion
        gray = 0.2989 * img(:, :, 1) + 0.5870 * img(:, :, 2) + 0.1140 * img(:, :, 3);
    else
        gray = img;
    end
endfunction

// ---------------------------------------------
// 3. Median Filter for Noise Removal (3x3 by default)
// ---------------------------------------------
function medImg = medianFilter(img, windowSize)
    if nargin < 2 then
        windowSize = 3;
    end
    [rows, cols] = size(img);
    pad = floor(windowSize/2);

    // Manual padding implementation
    padded = zeros(rows + 2*pad, cols + 2*pad);
    padded(pad+1:rows+pad, pad+1:cols+pad) = img;

    // Replicate padding (you can change this to other padding methods)
    padded(1:pad, pad+1:cols+pad) = ones(pad,1)*img(1,:);          // Top row
    padded(rows+pad+1:rows+2*pad, pad+1:cols+pad) = ones(pad,1)*img(rows,:);     // Bottom row
    padded(:, 1:pad) = ones(1,pad)*padded(:,pad+1);      // Left column
    padded(:, cols+pad+1:cols+2*pad) = ones(1,pad)*padded(:,cols+pad); // Right column


    medImg = zeros(rows, cols);
    for i = 1:rows
        for j = 1:cols
            window = padded(i:i+windowSize-1, j:j+windowSize-1);
            medImg(i, j) = median(window(:));
        end
    end
endfunction

// ---------------------------------------------
// 4. CLAHE: Contrast Limited Adaptive Histogram Equalization
// ---------------------------------------------
function enhanced = claheEnhance(img, clipLimit, tileGridSize)
    // Check that the image is grayscale
    if ndims(img) > 2 then
        error("CLAHE function expects a grayscale image.");
    end
    if clipLimit <= 0 then
        error("Clip limit must be positive.");
    end

    tileRows = tileGridSize(1);
    tileCols = tileGridSize(2);
    [rows, cols] = size(img);
    tileHeight = floor(rows / tileRows);
    tileWidth  = floor(cols / tileCols);
    enhanced = zeros(rows, cols);

    // Process each tile separately
    for i = 1:tileRows
        for j = 1:tileCols
            rStart = (i - 1) * tileHeight + 1;
            rEnd = min(i * tileHeight, rows);  // Ensure rEnd is not greater than rows

            cStart = (j - 1) * tileWidth + 1;
            cEnd = min(j * tileWidth, cols);  // Ensure cEnd is not greater than cols

            tile = img(rStart:rEnd, cStart:cEnd);
            [tileR, tileC] = size(tile);

            // Compute histogram with 256 bins
            hist = zeros(1, 256);
            for bin = 1:256
                lowerBound = (bin - 1) / 255;
                upperBound = bin / 255;
                hist(bin) = sum(tile(:) >= lowerBound & tile(:) < upperBound);
            end

            // Clip the histogram and redistribute excess counts
            excess = sum(max(hist - clipLimit, 0));
            hist = min(hist, clipLimit);
            hist = hist + excess / 256;

            // Compute cumulative distribution function (CDF)
            cdf = cumsum(hist);
            cdf = cdf / cdf($);  // Normalize so that maximum equals 1

            // Map the tile pixel values using the CDF
            tileEq = zeros(tileR, tileC);
            for a = 1:tileR
                for b = 1:tileC
                    intensity = tile(a, b);
                    index = floor(intensity * 255) + 1;
                    index = min(256, max(1, index));
                    tileEq(a, b) = cdf(index);
                end
            end

            enhanced(rStart:rEnd, cStart:cEnd) = tileEq;
        end
    end
endfunction

// ---------------------------------------------
// 5. Edge Detection Functions
// ---------------------------------------------

// 5a. Sobel Edge Detection
function sobelOut = sobelEdge(img)
    Gx = [ -1 0 1; -2 0 2; -1 0 1 ];
    Gy = [  1 2 1;  0 0 0; -1 -2 -1 ];
    sobelX = conv2(img, Gx, "same");
    sobelY = conv2(img, Gy, "same");
    sobelOut = sqrt(sobelX.^2 + sobelY.^2);
    sobelOut = sobelOut / max(sobelOut(:));
endfunction

// 5b. Prewitt Edge Detection
function prewittOut = prewittEdge(img)
    Gx = [ -1 0 1; -1 0 1; -1 0 1 ];
    Gy = [  1 1 1;  0 0 0; -1 -1 -1 ];
    gradX = conv2(img, Gx, "same");
    gradY = conv2(img, Gy, "same");
    prewittOut = sqrt(gradX.^2 + gradY.^2);
    prewittOut = prewittOut / max(prewittOut(:));
endfunction

// 5c. Canny Edge Detection

function edges = cannyEdge(img, lowThreshold, highThreshold)
    // Step 1: Gaussian smoothing (5x5 kernel, sigma = 1.4)
    gKernel = gaussianKernel(5, 1.4);
    smoothed = conv2(img, gKernel, "same");

    // Step 2: Compute gradients using Sobel operators
    Gx = [ -1 0 1; -2 0 2; -1 0 1 ];
    Gy = [  1 2 1;  0 0 0; -1 -2 -1 ];
    gradX = conv2(smoothed, Gx, "same");
    gradY = conv2(smoothed, Gy, "same");
    gradMag = sqrt(gradX.^2 + gradY.^2);
    gradDir = atan(gradY, gradX);  // use atan(y, x)

    // Step 3: Non-maximum suppression (simple approximation)
    [rows, cols] = size(img);
    nms = zeros(rows, cols);
    for i = 2:rows-1
        for j = 2:cols-1
            angle = gradDir(i,j);
            angleDeg = modulo(round(angle * 180 / %pi), 180);
            if ((angleDeg >= 0) & (angleDeg < 22.5)) | (angleDeg >= 157.5) then
                neighbor1 = gradMag(i, j-1);
                neighbor2 = gradMag(i, j+1);
            elseif (angleDeg >= 22.5) & (angleDeg < 67.5) then
                neighbor1 = gradMag(i-1, j+1);
                neighbor2 = gradMag(i+1, j-1);
            elseif (angleDeg >= 67.5) & (angleDeg < 112.5) then
                neighbor1 = gradMag(i-1, j);
                neighbor2 = gradMag(i+1, j);
            else
                neighbor1 = gradMag(i-1, j-1);
                neighbor2 = gradMag(i+1, j+1);
            end
            if gradMag(i,j) >= neighbor1 & gradMag(i,j) >= neighbor2 then
                nms(i,j) = gradMag(i,j);
            else
                nms(i,j) = 0;
            end
        end
    end

    // Step 4: Double thresholding and edge tracking by hysteresis
    strong = nms >= highThreshold;
    weak = (nms >= lowThreshold) & (nms < highThreshold);
    edges = zeros(rows, cols);
    edges(strong) = 1;
    changed = 1;
    while changed
        oldEdges = edges;
        for i = 2:rows-1
            for j = 2:cols-1
                if weak(i,j) & (max(max(edges(i-1:i+1, j-1:j+1))) == 1) then
                    edges(i,j) = 1;
                end
            end
        end
        if norm(edges - oldEdges) < 1e-6 then
            changed = 0;
        end
    end
    edges = edges > 0;
endfunction

// ---------------------------------------------
// 6a. Gaussian Kernel Generator (for Canny)
// ---------------------------------------------
function kernel = gaussianKernel(s, sigma)
    half = floor(s/2);
    [X, Y] = ndgrid(-half:half, -half:half);
    kernel = exp(-(X.^2 + Y.^2) / (2 * sigma^2));
    kernel = kernel / sum(kernel(:));
endfunction

// ---------------------------------------------
// 7. Global Threshold Segmentation
// ---------------------------------------------
function binaryImg = thresholdSegmentation(img, T)
    // For each pixel: if f(x,y) > T then 1, else 0.
    binaryImg = img > T;
endfunction

// ---------------------------------------------
// 8. Custom k-Means Clustering Function for 1D Data
// ---------------------------------------------
function [clusters, centers] = myKmeans(data, k, maxIter)
    if nargin < 3 then
        maxIter = 100;
    end
    n = length(data);
    clusters = zeros(n, 1);
    rndIdx = floor(1 + (n - 1) * rand(1, k));
    centers = data(rndIdx);
    for iter = 1:maxIter
        oldCenters = centers;
        for i = 1:n
            dists = abs(data(i) - centers);
            [dummy, idx] = min(dists);
            clusters(i) = idx;
        end
        for j = 1:k
            indices = find(clusters == j);
            if length(indices) > 0 then
                centers(j) = sum(data(indices)) / length(indices);
            else
                centers(j) = data(floor(1 + (n - 1) * rand()));
            end
        end
        if norm(centers - oldCenters) < 1e-6 then
            break;
        end
    end
endfunction

// ---------------------------------------------
// 9. k-Means Clustering for Tumor Segmentation (Wrapper)
// ---------------------------------------------
function segmented = kmeansSegment(img, k)
    [rows, cols] = size(img);
    data = matrix(img, rows * cols, 1);
    [clusters, centers] = myKmeans(data, k);
    segmented = matrix(clusters, rows, cols);
endfunction

// ---------------------------------------------
// 10. Main Brain Tumor Detection Pipeline
// ---------------------------------------------
function mainTumorDetectionPipeline()
    disp("Starting Brain Tumor Detection Pipeline...");

    // Step 1: Load the original MRI image
    try
        originalImg = loadImageHardCoded();
    catch err
        disp("Error loading image: " + err.message);
        return;
    end

    // Step 2: Convert to Grayscale (if needed)
    grayImg = rgbToGray(originalImg);

    // Step 3: Noise Removal using Median Filter (as per paper)
    denoisedImg = medianFilter(grayImg, 3);

    // Step 3a: Segmentation after Median Filter (Thresholding)
    medianThreshImg = thresholdSegmentation(denoisedImg, 0.6); // Adjust threshold as needed

    // Step 4: Image Enhancement using CLAHE (as in original code)
    enhancedImg = claheEnhance(denoisedImg, 20, [8, 8]);

    // Step 5: Edge Detection (as per paper, using Sobel, Prewitt and Canny)
    sobelImg    = sobelEdge(enhancedImg);
    prewittImg  = prewittEdge(enhancedImg);
    cannyImg    = cannyEdge(enhancedImg, 0.05, 0.15);

    // Step 6: Segmentation (as per paper, k-means after thresholding)
    // 6a. Global Thresholding (choose T between 0 and 1, e.g., 0.5)
    threshImg = thresholdSegmentation(enhancedImg, 0.5);

     // Step 6b: Segmentation after Global Thresholding (k-means)
    threshKmeansImg = kmeansSegment(threshImg, 2); // k=2 to separate tumor from background

    // 6b. k-Means Clustering (e.g., k = 4)
    kmeansImg = kmeansSegment(enhancedImg, 4);

    // Step 7: Final Predicted Tumor (based on k-means result)
    predictedTumor = kmeansImg > 2; // Assuming tumor is one of the higher intensity clusters.

    // Display Results in a 4x3 grid
    scf(0);
    clf();
    // Row 1: Original, Grayscale, Denoised
    subplot(4, 3, 1);
    imshow(originalImg);
    title("Original Image");

    subplot(4, 3, 2);
    imshow(grayImg);
    title("Grayscale Image");

    subplot(4, 3, 3);
    imshow(denoisedImg);
    title("Median Filtered");

    // Row 2: Median Threshold, Enhanced, Sobel
     subplot(4, 3, 4);
    imshow(medianThreshImg);
    title("Segmentation after Median");

    subplot(4, 3, 5);
    imshow(enhancedImg);
    title("CLAHE Enhanced");

    subplot(4, 3, 6);
    imshow(sobelImg);
    title("Sobel Edge");

    // Row 3: Prewitt, Canny, Threshold Segmentation
    subplot(4, 3, 7);
    imshow(prewittImg);
    title("Prewitt Edge");

    subplot(4, 3, 8);
    imshow(cannyImg);
    title("Canny Edge");

    subplot(4, 3, 9);
    imshow(threshImg);
    title("Global Thresholding");

    // Row 4: Threshold k-means, k-Means Clustering, Final Predicted Tumor
    subplot(4, 3, 10);
    imshow(threshKmeansImg / max(threshKmeansImg(:))); // Normalize for display
    title("Segmentation after Thresholding");

    subplot(4, 3, 11);
    imshow(kmeansImg / max(kmeansImg(:))); // Normalize for display
    title("k-Means Clustering");

    subplot(4, 3, 12);
    imshow(predictedTumor);
    title("Final Predicted Tumor");

    disp("Brain Tumor Detection Pipeline Complete.");
endfunction

// ---------------------------------------------
// Measure Execution Time
// ---------------------------------------------
tic();  // Start timer

// Run the Main Brain Tumor Detection Pipeline
mainTumorDetectionPipeline();

elapsedTime = toc();  // Stop timer
disp("Total Execution Time: " + string(elapsedTime) + " seconds");
