// Engine Specifications
D = 0.1;          // Cylinder bore [m]
stroke = 0.124;   // Stroke length [m]
V_d = %pi/4*D^2*stroke; // Displacement volume [m³]
comp_ratio = 18;  // Compression ratio
V_c = V_d/(comp_ratio-1); // Clearance volume [m³]
RPM = 1500;       // Engine speed [rev/min]
N = RPM/60;       // Revolutions per second [Hz]

// Valve Timing (Section VI.b)
IVO = -240;       // Inlet valve open [deg]
IVC = 480;        // Inlet valve close [deg ABDC]
EVO = 130;        // Exhaust valve open [deg BTDC]
EVC = 10;         // Exhaust valve close [deg ATDC]


// Thermodynamic Initialization
theta = [-180:1:180]'; // Crank angle vector [deg] - made column vector
n_theta = length(theta);

// Preallocate arrays
V = zeros(n_theta,1);
P = zeros(n_theta,1);
T = zeros(n_theta,1);

// Initial conditions
P(1) = 1.0e5;    // Pressure at IVC [Pa]
T(1) = 300;      // Temperature at IVC [K]
gamma = 1.4;     // Specific heat ratio

// Combustion Parameters
Q_LHV = 42e6;     // Lower heating value [J/kg]
AF_ratio = 18;    // Air-fuel ratio
comb_start = -10; // Start of combustion [deg ATDC]
comb_dur = 40;    // Combustion duration [deg]
a_wiebe = 5;      // Wiebe efficiency factor
n_wiebe = 3;      // Wiebe form factor

// Heat Transfer Parameters (Section V.c)
Tw = 400;         // Wall temperature [K]
h_conv = 500;     // Convective heat transfer coeff [W/m²K]
C_rad = 2.81e-8;  // Radiation constant [W/m²K⁴]

// Function to calculate specific heat capacity
function cp = Cp(T)
    cp = 1004.5; // Simplified constant value for air
endfunction

// Function to calculate valve flow (placeholder)
function [dm_in, dm_out] = valveFlow(P, T, t)
    dm_in = 0;
    dm_out = 0;
endfunction

// Engine ODE Function
function dy = engineODE(t, y)
    P = y(1);
    T = y(2);
    V = y(3);
    
    // Cylinder volume calculation (Eq.9-10)
    V = V_c + (V_d/2)*(1 - cosd(t));
    dVdt = (V_d/2) * sind(t) * (%pi/180);
    
    // Phase detection
    if t >= IVC & t < EVO then
        // Closed phase calculations
        if t >= comb_start & t <= (comb_start+comb_dur) then
            x_b = 1 - exp(-a_wiebe*((t-comb_start)/comb_dur)^n_wiebe);
            dQ_comb = 0.1*Q_LHV*x_b;
        else
            dQ_comb = 0;
        end
        
        // Heat transfer calculation (Eq.13)
        A = %pi*D^2/4 + %pi*D*(V_c/V_d)*stroke;
        q_conv = h_conv*A*(T - Tw);
        q_rad = C_rad*A*(T^4 - Tw^4);
        dQ_w = q_conv + q_rad;
        
        // Energy equation (Eq.2)
        m = P*V/(287*T);
        dTdt = (dQ_comb - P*dVdt - dQ_w)/(m*717);
        dPdt = (m*287*dTdt - P*dVdt)/V;
    else
        // Open phase calculations
        dQ_w = 0;  // Initialize heat transfer
        [dm_in, dm_out] = valveFlow(P, T, t);
        m = P*V/(287*T);  // Mass calculation
        
        // Pressure differential equation (Eq.21)
        dPdt = (gamma-1)/V * (dm_in*Cp(T)*300 - dm_out*Cp(T)*T - (P*dVdt) - dQ_w);
        dTdt = (dPdt*V + P*dVdt)/(m*287);
    end
    
    dy = [dPdt; dTdt; dVdt];  // Return as column vector
endfunction

// ========================
// Solver Setup and Execution
// ========================
t = linspace(theta(1), theta($), n_theta);

// Function wrapper for ODE solver
function dy = f(t, y)
    dy = engineODE(t, y);
endfunction

// Initial conditions
y0 = [P(1); T(1); V_c + (V_d/2)*(1 - cosd(theta(1)))];

// Solve using Runge-Kutta method
y = ode("rk", y0, t(1), t, f);

// Extract and format results
y = y';  // Transpose to get correct dimensions
P = y(:,1);  // Pressure vector
T = y(:,2);  // Temperature vector
V = y(:,3);  // Volume vector


// Performance Calculations
// Indicated Mean Effective Pressure
imep = inttrap(theta, P.*diff([V;V($)])) / V_d;

// Power and Torque
power = imep * V_d * N / 2;
torque = power/(2*%pi*N);

// Display Results
mprintf('\nPerformance Results:\n');
mprintf('Indicated Power: %.2f kW\n', power/1e3);
mprintf('Engine Torque: %.2f N·m\n', torque);
mprintf('IMEP: %.2f bar\n', imep/1e5);


// Visualization
f = scf(1); clf(f);
plot(theta, P/1e5, 'linewidth', 2);
xlabel('Crank Angle [deg]');
ylabel('Pressure [bar]');
title('In-Cylinder Pressure Development');
xgrid;

//Add second plot for temperature
f = scf(2); clf(f);
plot(theta, T, 'linewidth', 2);
xlabel('Crank Angle [deg]');
ylabel('Temperature [K]');
title('In-Cylinder Temperature Development');
xgrid;

dVdt = (V_d/2) * sind(theta) * (%pi/180);  // [m³/deg]

//Volume Rate Change vs Crank Angle
scf(3); clf();
plot(theta, dVdt*1e6, 'g-', 'LineWidth', 2);  // Convert to liters/deg
xlabel('Crank Angle [deg]', 'fontsize', 3);
ylabel('dV/dθ [L/deg]', 'fontsize', 3);
title('Volume Rate Change vs Crank Angle', 'fontsize', 4);
xgrid();

// Combined Thermodynamic Parameters
scf(4); clf();
subplot(3,1,1);
plot(theta, P/1e5, 'b-', 'LineWidth', 1.5);
ylabel('Pressure [bar]');
title('Combined Thermodynamic Parameters');

subplot(3,1,2);
plot(theta, T, 'r-', 'LineWidth', 1.5);
ylabel('Temperature [K]');

subplot(3,1,3);
plot(theta, dVdt*1e6, 'g-', 'LineWidth', 1.5);
xlabel('Crank Angle [deg]');
ylabel('dV/dθ [L/deg]');

//Combustion Analysis (Wiebe Function Implementation)
scf(5); clf();
x_b = zeros(theta);
dQ_comb = zeros(theta);
for k = 1:length(theta)
    if theta(k) >= comb_start & theta(k) <= comb_start+comb_dur
        x_b(k) = 1 - exp(-a_wiebe*((theta(k)-comb_start)/comb_dur)^n_wiebe);
        dQ_comb(k) = 0.1*Q_LHV*x_b(k);
    end
end

plot(theta, x_b, 'm-', theta, dQ_comb/1e3, 'c--', 'LineWidth', 2);
xlabel('Crank Angle [deg]', 'fontsize', 3);
ylabel('Combustion Parameters', 'fontsize', 3);
title('Combustion Progress (Wiebe Function)', 'fontsize', 4);
legend(['Burn Fraction'; 'Heat Release (kJ/deg)']);
xgrid();
