// ===== 1. Monte Carlo Sampling =====
function X = monte_carlo(n, N)
    X = rand(N, n);
endfunction

// ===== 2. Latin Hypercube Sampling (LHS) =====
function X = lhs(n, N)
    X = zeros(N, n);
    for j = 1:n
        perm = gsort(rand(N,1), "g", "i"); // Emulate randperm
        X(:, j) = ((perm - 1) + rand(N, 1)) / N;
    end
endfunction

// ===== 3. Test Function: Type A Function 1A =====
function y = f1A(x)
    [N, n] = size(x);
    y = zeros(N,1);
    for i = 1:n
        term = (-1)^i * cumprod(x(:,1:i), 2);
        y = y + term(:, i);
    end
endfunction

// ===== 4. Error and Convergence Calculation =====
function [N_values, errors, alphas] = compute_errors(func, true_val, n, K)
    methods = list("MC", "LHS");
    N_values = round(logspace(2, 4, 10)); // From 100 to 10000
    errors = zeros(length(N_values), length(methods));

    for m = 1:length(methods)
        for i = 1:length(N_values)
            N = N_values(i);
            errs = zeros(K,1);

            for k = 1:K
                method = methods(m);
                select method
                    case "MC"
                        X = monte_carlo(n, N);
                    case "LHS"
                        X = lhs(n, N);
                end

                y = func(X);
                est = mean(y);
                errs(k) = abs(true_val - est);
            end

            errors(i, m) = sqrt(mean(errs.^2));
        end
    end

    // Compute convergence rate alpha from log-log slope
    alphas = zeros(length(methods), 1);
    for m = 1:length(methods)
        p = polyfit(log(N_values)', log(errors(:,m)), 1);
        alphas(m) = -p(1);
    end
endfunction

// ===== 5. Convergence Plotting Function =====
function plot_convergence(N_values, errors, alphas, title_str)
    scf(0);
    clf();
    loglog(N_values, errors(:,1), 'b-x', "LineWidth", 2);
    loglog(N_values, errors(:,2), 'g-o', "LineWidth", 2);

    legend([
        "MC (α = " + msprintf("%.2f", alphas(1)) + ")";
        "LHS (α = " + msprintf("%.2f", alphas(2)) + ")"
    ], "in_upper_right");

    xtitle(title_str, "Number of Samples (log scale)", "RMSE (log scale)");
    xgrid();
endfunction

// ===== 6. Main Execution =====
true_val = -1/3 * (1 - (-0.5)^5); // True value for f1A with n=5
[N_values, errors, alphas] = compute_errors(f1A, true_val, 5, 30);
plot_convergence(N_values, errors, alphas, "Function 1A - MC vs LHS Convergence");

show_window(); // Show final plot
