funcprot(0); // avoid warning on function redefinition

// Custom randperm
function p = my_randperm(n)
    p = 1:n;
    for i = n:-1:2
        j = ceil(rand() * i);
        tmp = p(i);
        p(i) = p(j);
        p(j) = tmp;
    end
endfunction

// Latin Hypercube Sampling
function lhs = lhsample(n, d)
    lhs = zeros(n, d);
    for j = 1:d
        perm = my_randperm(n);
        lhs(:, j) = (perm' + rand(n, 1) - 1) / n;
    end
endfunction

// Inverse Normal CDF approximation (Acklam's method)
function x = invnorm_approx(p)
    a = [ -3.969683028665376e+01, 2.209460984245205e+02, -2.759285104469687e+02, ...
          1.383577518672690e+02, -3.066479806614716e+01, 2.506628277459239e+00 ];
    b = [ -5.447609879822406e+01, 1.615858368580409e+02, -1.556989798598866e+02, ...
          6.680131188771972e+01, -1.328068155288572e+01 ];
    c = [ -7.784894002430293e-03, -3.223964580411365e-01, -2.400758277161838e+00, ...
         -2.549732539343734e+00, 4.374664141464968e+00, 2.938163982698783e+00 ];
    d = [  7.784695709041462e-03, 3.224671290700398e-01, 2.445134137142996e+00, ...
           3.754408661907416e+00 ];

    plow  = 0.02425;
    phigh = 1 - plow;

    x = zeros(p);
    for i = 1:length(p)
        if p(i) < plow then
            q = sqrt(-2 * log(p(i)));
            x(i) = (((((c(1)*q + c(2))*q + c(3))*q + c(4))*q + c(5))*q + c(6)) / ...
                   ((((d(1)*q + d(2))*q + d(3))*q + d(4))*q + 1);
        elseif p(i) <= phigh then
            q = p(i) - 0.5;
            r = q * q;
            x(i) = (((((a(1)*r + a(2))*r + a(3))*r + a(4))*r + a(5))*r + a(6))*q / ...
                   (((((b(1)*r + b(2))*r + b(3))*r + b(4))*r + b(5))*r + 1);
        else
            q = sqrt(-2 * log(1 - p(i)));
            x(i) = -(((((c(1)*q + c(2))*q + c(3))*q + c(4))*q + c(5))*q + c(6)) / ...
                     ((((d(1)*q + d(2))*q + d(3))*q + d(4))*q + 1);
        end
    end
endfunction

// Main script
n = 1000;
lhs_unit = lhsample(n, 2);

// Prevent exact 0 or 1
epsilon = 1e-10;
p1 = min(max(lhs_unit(:, 1), epsilon), 1 - epsilon);
z = invnorm_approx(p1); // inverse standard normal
v1 = 1.0 + 0.5 * z;      // N(1.0, 0.5)

v2 = 2.0 + lhs_unit(:, 2); // U(2.0, 3.0)

// Plotting
scf();
plot(v1, v2, '.');
xtitle("Latin Hypercube Sampling", "Variable #1: Normal(1.0,0.5)", "Variable #2: Uniform(2,3)");





// Define Halton sequence generator
function h = halton(n, d)
    primes = [2, 3, 5, 7, 11, 13, 17, 19]; // primes for each dimension
    h = zeros(n, d);
    for j = 1:d
        base = primes(j);
        for i = 1:n
            f = 1;
            r = 0;
            k = i;
            while k > 0
                f = f / base;
                r = r + f * modulo(k, base);
                k = floor(k / base);
            end
            h(i, j) = r;
        end
    end
endfunction

// Generate Halton sequence
N = 256;
pts = halton(N, 2);

// Define Q(t1, t2)
t1 = 0.6;
t2 = 0.45;

// Count how many points are in the shaded rectangle
inside = find(pts(:,1) <= t1 & pts(:,2) <= t2);
NQ = length(inside);
disp("NQ = " + string(NQ));

// Open figure
scf(0);
clf();

// Draw shaded rectangle using patch
x = [0, t1, t1, 0];
y = [0, 0, t2, t2];
xrects([0; 0; t1; t2], color("lightblue"));  // proper column vector

// Plot all points
plot(pts(:,1), pts(:,2), 'r.');

// Mark the Q(t1, t2) point
plot(t1, t2, 'bd'); // blue diamond

// Draw the rectangle border
xpoly([0, t1, t1, 0, 0], [0, 0, t2, t2, 0], "lines");

// Add title and axis labels
xtitle("2D LDS Sampling - Local Discrepancy", "t₁", "t₂");
