// Pathloss vs. UAV Height Simulation 
clc;
clear;
close;

// Parameters
num_users = 1000; // Number of users
Coverage_area = 1000; // Size of the area (1000m x 1000m)
frequency = 2e9; // Frequency in Hz (2 GHz)
c = 3e8; // Speed of light in m/s
sigma = 0;// Shadowing standard deviation (dB)

// Environmental Parameters
a1 = [0.1 1 1.6 2.3]; 
b1 = [21 20 23 34];
alpha1 = [5.2 10.6 11.95 26.5]; 
beta1 = [0.35 0.18 0.14 0.13];

// Generate user positions (randomly distributed)
x_users = rand(1, num_users) * Coverage_area - Coverage_area / 2; // x-coordinates in range [-500, 500]
y_users = rand(1, num_users) * Coverage_area - Coverage_area / 2; // y-coordinates in range [-500, 500]

// Loop through each altitude
uav_altitudes = 0:20:1000; // Altitudes from 1m to 1000m
Pathloss = zeros(1, length(uav_altitudes)); // Preallocate for pathloss values

for i = 1:4
    eta_los = 10 * log10(a1(i)); 
    eta_Nlos = 10 * log10(b1(i)); // Excessive path loss 
    for j = 1:length(uav_altitudes)
        altitude = uav_altitudes(j);
        // Calculate distances from UAV to users
        distances = sqrt(x_users.^2 + y_users.^2 + altitude^2); // 3D distance
        A = eta_los - eta_Nlos;
        B = 10 * log10(distances) + 20 * log10((4 * %pi * frequency) / c) + eta_Nlos;
        shadowing = grand(1, 1, "nor", 0, sigma);  // Random shadowing
        theta = (180 / %pi) * atan(altitude ./ sqrt(x_users.^2 + y_users.^2));
        P_L1 = A ./ (1 + (alpha1(i) * exp(-beta1(i) * (theta - alpha1(i)))));   
        P_L = P_L1 + B+shadowing;
        Pathloss(j) = mean(P_L); // Average pathloss across users
    end
    // Colors for different environments
    colors = ["b", "r", "g", "m"]; // Blue, Red, Green, Magenta
    plot(uav_altitudes, Pathloss, colors(i) , "LineWidth", 2); // Plot with specified color and markers
end


// Add legend and labels
legends = ["Suburban", "Urban", "Dense Urban", "Highrise Urban"];
legend(legends, "Location", "northwest");
xlabel("UAV Altitude (m)");
ylabel("PowerLoss (dB)");
title("PowerLoss vs. UAV Altitude");
xgrid

