// Grid and Simulation Parameters
nx = 120; // Grid points in x-direction
ny = nx; // Grid points in y-direction
iterations = 2000; // Number of iterations

// Geometry Initialization
geometry = ones(ny, nx); // Wall cells (geometry=1)
geometry(2:ny-1, 2:nx-1) = 0; // Fluid cells (geometry=0)
geometry(1, :) = 2; // Moving lid cells (geometry=2)

rho_0 = 1; // Initial density
u_0 = 0.1; // Lid velocity

Re = 1000; // Reynolds number
viscosity = (ny-1)*u_0/Re; // Kinematic viscosity
tau = (6*viscosity + 1) / 2; // Relaxation time

// Lattice Directions (D2Q9)
C = 1; E = 2; S = 3; W = 4; N = 5; NE = 6; SE = 7; SW = 8; NW = 9;
// Direction indices mapping:
// 9 5 6
// 4 1 2
// 8 3 7

// Lattice Distribution Functions
f = zeros(nx*ny, 9);
feq = zeros(nx*ny, 9);
rho = zeros(nx*ny, 1);
ux = zeros(nx*ny, 1);
uy = zeros(nx*ny, 1);
usqr = zeros(nx*ny, 1);

// Initialize Distributions
f(:, C) = rho_0 * 4 / 9;
f(:, [E, S, W, N]) = rho_0 / 9;
f(:, [NE, SE, SW, NW]) = rho_0 / 36;

// Identify Fluid, Wall, and Lid Cells
FL = find(geometry == 0);
WALL = find(geometry == 1);
DR = find(geometry == 2);

// LBM Iterations
for i = 1:iterations
    // Compute Macroscopic Variables
    rho(:) = sum(f, 2);
    ux(:) = (f(:, E) - f(:, W) + f(:, NE) + f(:, SE) - f(:, SW) - f(:, NW)) ./ rho;
    uy(:) = (f(:, N) - f(:, S) + f(:, NE) + f(:, NW) - f(:, SE) - f(:, SW)) ./ rho;
    
    ux(DR) = u_0; // Set lid velocity
    uy(DR) = 0;
    usqr(:) = ux .* ux + uy .* uy;
    
    // Equilibrium Distribution Function
    feq(:, C) = (4 / 9) * rho .* (1 - 1.5 * usqr);
    feq(:, E) = (1 / 9) * rho .* (1 + 3 * ux + 4.5 * ux .^ 2 - 1.5 * usqr);
    feq(:, S) = (1 / 9) * rho .* (1 - 3 * uy + 4.5 * uy .^ 2 - 1.5 * usqr);
    feq(:, W) = (1 / 9) * rho .* (1 - 3 * ux + 4.5 * ux .^ 2 - 1.5 * usqr);
    feq(:, N) = (1 / 9) * rho .* (1 + 3 * uy + 4.5 * uy .^ 2 - 1.5 * usqr);
    feq(:, NE) = (1 / 36) * rho .* (1 + 3 * (ux + uy) + 4.5 * (ux + uy) .^ 2 - 1.5 * usqr);
    feq(:, SE) = (1 / 36) * rho .* (1 + 3 * (ux - uy) + 4.5 * (ux - uy) .^ 2 - 1.5 * usqr);
    feq(:, SW) = (1 / 36) * rho .* (1 + 3 * (-ux - uy) + 4.5 * (-ux - uy) .^ 2 - 1.5 * usqr);
    feq(:, NW) = (1 / 36) * rho .* (1 + 3 * (-ux + uy) + 4.5 * (-ux + uy) .^ 2 - 1.5 * usqr);

    // Bounce-Back Boundary Conditions
    f(WALL, [C, E, S, W, N, NE, SE, SW, NW]) = f(WALL, [C, W, N, E, S, SW, NW, NE, SE]);

    // Moving Wall Conditions
    f(DR, :) = feq(DR, :);

    // Collision Step
    f(FL, :) = f(FL, :) * (1 - 1 / tau) + feq(FL, :) / tau;

    // Propagation Step
    f = matrix(f, ny, nx, 9);
    f(:, 2:nx, E) = f(:, 1:nx-1, E);
    f(2:ny, :, S) = f(1:ny-1, :, S);
    f(:, 1:nx-1, W) = f(:, 2:nx, W);
    f(1:ny-1, :, N) = f(2:ny, :, N);
    f(1:ny-1, 2:nx, NE) = f(2:ny, 1:nx-1, NE);
    f(2:ny, 2:nx, SE) = f(1:ny-1, 1:nx-1, SE);
    f(2:ny, 1:nx-1, SW) = f(1:ny-1, 2:nx, SW);
    f(1:ny-1, 1:nx-1, NW) = f(2:ny, 2:nx, NW);
    f = matrix(f, nx*ny, 9);
end

// Compute velocity magnitude
u = sqrt(ux .^ 2 + uy .^ 2) / u_0;
u = matrix(u, ny, nx);

// Plot using `surf` function
surf(u);
colormap(jet(50));  // Set the colormap with 50 levels

// Add a colorbar
colorbar();  // Simply call `colorbar()` without arguments

// Set title and axis properties
title("Relative macroscopic velocity magnitude (u/u_0) after " + string(iterations) + " iterations");
ax = gca();  // Get current axis
ax.data_bounds = [0, 0; nx, ny];  // Set data bounds
ax.box = "on";  // Show box around the plot

// End of Code
