// Hello! please ensure that IPCV, ANN toolbox and Scilab Neural Network Module are loaded
// Clear environment
clear;

//Please give the same vaues to both the following functions for consistency
rand("seed", 299)

function out=%l_zeros(a)
  // Assumes 'a' is a list of numeric size dimensions, or list wrapping a size vector
  // Convert list to numeric vector
  dims = [];
  for k = 1:length(a)
    elem = a(k);
    if typeof(elem) == "constant" then
      dims(k) = elem;
    else
      error("List element %" + string(k) + " is not numeric.");
    end
  end
  // Now call builtin zeros with these dims
  out = zeros(dims);
endfunction

// Define main classes and their subfolders
main_classes = ["Hazardous", "Non-Recyclable", "Organic", "Recyclable"];
subfolders = list( ...
    ["batteries", "e-waste", "pesticides", "paints"], ...
    ["ceramic_product", "diapers", "platics_bags_wrappers", "sanitary_napkin", "stroform_product"], ...
    ["coffee_tea_bags", "egg_shells", "food_scraps", "kitchen_waste", "yard_trimmings"], ...
    ["plastic_bottles", "paper_products", "glass_containers", "cans_all_type"] ...
);
base_path = "C:\Users\Naini Diwan\Downloads\Recyclable and Non Recyclable Waste - Copy";

// Initialize feature and label arrays
X = [];
Y = [];
function feat=extract_features_ipcv(img)
    img = imresize(img, [64, 64]);
    dims = size(img);

    // Ensure RGB format
    if length(dims) == 2 then
        // Grayscale → replicate to RGB
        img = cat(3, img, img, img);
    end

    feat = [];
    for ch = 1:3
        h = imhist(img(:,:,ch), 48);
        feat = [feat, h']; // 1×48 row vector for each channel
    end
endfunction

n_classes = size(main_classes, "*");
disp("Depending upon size of the input dataset, please wait for feature extraction from the images - this may take time")
for i = 1:n_classes
    class_path = base_path + "\" + main_classes(i);
    subfolder_list = subfolders(i);
    n_subfolders = size(subfolder_list, "*");
    for j = 1:n_subfolders
        sub_path = class_path + "\" + subfolder_list(j);
        files = ls(sub_path);
        for k = 1:size(files, 1)
            img = imread(sub_path + "\" + files(k));
            feat = extract_features_ipcv(img);
            if size(X, 1) == 0 then
                X = feat;
            else
                X = [X; feat];
            end
            Y = [Y; i];
        end
    end
end
disp("Labels and feature matrix made")

n = size(X, 1);
[dummy, perm] = gsort(rand(n, 1), "g", "i"); // Random permutation
split = round(0.75 * n);

X_train = X(perm(1:split), :);
Y_train = Y(perm(1:split));
X_test  = X(perm(split+1:n), :);
Y_test  = Y(perm(split+1:n));

// --- Powerful Network Architecture: 2 hidden layers, LM training ---
num_features = size(X_train, 2);
num_classes = 4;
net_arch_complex = [num_features, 35, 15, num_classes]; // 2 hidden layers, 4 outputs
af_complex = ['ann_tansig_activ', 'ann_tansig_activ', 'ann_purelin_activ'];

// Prepare targets as one-hot encoding (num_classes x num_samples)
T_train = zeros(num_classes, size(Y_train,1));
for i=1:size(Y_train,1)
    if Y_train(i) >= 1 & Y_train(i) <= num_classes then
        T_train(Y_train(i), i) = 1;
    else
        error("Invalid class label: " + string(Y_train(i)));
    end
end

if size(T_train,2) ~= size(X_train,1) then
    T_train = T_train';
end

P = X_train';
T = T_train;
dt = getdate();
mprintf("\n[You started at time %02d:%02d:%02d] \n", dt(7), dt(8), dt(9));
disp("------------------------------------- Training the Network Architecture -----------------------------------");
disp("The default dataset takes approximately 3 hours to process")
tic();
W_best = ann_FFBP_lm(P, T, net_arch_complex, af_complex);
train_time = toc();
printf("Training completed in %.2f seconds.\n", train_time);


//========================================================================//
//====================== INTERACTIVE PREDICTION MENU =====================//
//========================================================================//

disp("Training complete. What would you like to do next?");

disp("1 - Evaluate on the 25% test set");
disp("2 - Predict on a new image/folder");
choice = input("Enter 1 or 2: ");

// --- Option 1: Evaluate on the original 25% test set ---
// You will get to see pie chart for composition of the dataset, bar chart for accuracies of various classes and a PCA scatter plot, along with prediction of a randomly selected image
if choice == 1 then
    disp("Performing evaluation on the 25% test set...");

    // Prediction
    Y_pred = ann_FFBP_run(X_test', W_best, af_complex);
    [foo, Y_pred_class] = max(Y_pred, 'r');
    Y_pred_class = Y_pred_class';
    disp("Prediction and Evaluation Successfully Completed")

    // Confusion matrix
    conf_mat = zeros(4,4);
    for i=1:length(Y_test)
        if Y_test(i) >= 1 & Y_test(i) <= 4 & Y_pred_class(i) >= 1 & Y_pred_class(i) <= 4 then
            conf_mat(Y_test(i), Y_pred_class(i)) = conf_mat(Y_test(i), Y_pred_class(i)) + 1;
        else
            disp("Warning: Invalid class index at position " + string(i));
        end
    end

    // --- Visualisation ---
    // a. Pie Chart
    scf(2);clf;
    class_counts = [sum(Y==1), sum(Y==2), sum(Y==3), sum(Y==4)];
    pie(class_counts, main_classes);
    xtitle('Class Distribution in Dataset');

    // b. Bar Chart
    scf(3);
    acc_per_class = diag(conf_mat) ./ sum(conf_mat, 2);
    bar(acc_per_class, 'r');
    xtitle('Accuracy per Class');
    ax = gca();
    ax.x_ticks.locations = [1;2;3;4];
    ax.x_ticks.labels = matrix(main_classes, -1, 1);

    // c. PCA Scatter Plot
    scf(4); clf;
    if size(X,1) < 3 | size(X,2) < 2 then
        disp("Insufficient data for PCA visualization. Skipping.");
    else
        mu = mean(X, "r");
        centered_X = X - repmat(mu, size(X,1), 1);
        [U, S, V] = svd(centered_X);
        X_pca = centered_X * V(:, 1:2);
        colors = [1 0 0; 0 0 1; 0 1 0; 1 1 0];
        handles = [];
        for cls = 1:4
            idx = find(Y == cls);
            if ~isempty(idx) then
                scatter(X_pca(idx,1), X_pca(idx,2), 5);
                h = gce();
                poly = h.children(1);
                poly.mark_mode = "on";
                poly.mark_style = 9;
                poly.mark_size = 4;
                poly.mark_foreground = color(colors(cls,1)*255, colors(cls,2)*255, colors(cls,3)*255);
                handles(cls) = h;
            end
        end
        xtitle('Feature Space Visualization (PCA)', 'PC1', 'PC2');
        ax = gca();
        ax.auto_scale = "on";
        legend(handles, main_classes, 3);
        drawnow();
    end

    // d. Sample Image Display
    scf(5);
    sample_idx = grand(1, 1, "uin", 1, length(Y_test));
    original_idx = perm(split + sample_idx);
    img_counter = 0;
    found = %f;
    for i = 1:n_classes
        class_path = base_path + "\" + main_classes(i);
        subfolder_list = subfolders(i);
        for j = 1:size(subfolder_list, "*")
            sub_path = class_path + "\" + subfolder_list(j);
            files = ls(sub_path);
            for k = 1:size(files, 1)
                img_counter = img_counter + 1;
                if img_counter == original_idx then
                    sample_img = imread(sub_path + "\" + files(k));
                    found = %t;
                    break;
                end
            end
            if found then break; end
        end
        if found then break; end
    end
    if found then
        imshow(sample_img);
        xtitle("Predicted class of sample image: " + main_classes(Y_pred_class(sample_idx)));
    else
        disp("Error: Could not locate original image.");
    end

// --- Option 2: Predict on new image(s) from user ---
elseif choice == 2 then
    disp("You chose to predict on a new image or a folder of images.");

    // Function to process and predict a single image
    function predict_and_display(image_path, trained_weights, activation_funcs, class_names)
        try
            // Read and extract features from the image
            img = imread(image_path);
            feat = extract_features_ipcv(img);

            // Run the prediction
            Y_pred = ann_FFBP_run(feat', trained_weights, activation_funcs);
            [foo, Y_pred_class] = max(Y_pred, 'r');
            predicted_class_name = class_names(Y_pred_class);

            // Display the image and its predicted class
            scf(); // Create a new figure window for each image
            imshow(img);
            xtitle("Predicted Class: " + predicted_class_name);
        catch
            disp("Could not process file with path- " + image_path);
            disp("It might not be a valid image file or the path is incorrect.");
        end
    endfunction

    // Get the path from the user
    target_path = input("Enter the full path to an image file OR a folder of images (without quotes)- ", "string");

    // Check if the path is a single file or a directory
    if isfile(target_path) then
        // User provided a path to a single file
        disp("Processing the image...");
        predict_and_display(target_path, W_best, af_complex, main_classes);

    elseif isdir(target_path) then
        // User provided a path to a folder
        disp("Processing all images in the folder...");
        files = ls(target_path);

        if isempty(files) then
            disp("The selected folder is empty or contains no valid files.");
        else
            for i = 1:size(files, 1)
                // Construct the full path for each file
                file_path = target_path + "\" + files(i);
                if isfile(file_path) then
                    predict_and_display(file_path, W_best, af_complex, main_classes);
                end
            end
        end
    else
        // The provided path is not valid
        error("The path provided is not a valid file or directory. Please run the script again.");
    end

else
    // User closed the menu without choosing
    disp("No valid option selected. Exiting.");
end
