clc;
clear;
funcprot(0);

global img_handle gray_handle seg_handle thr_handle original_img;

// =======================
// K-MEANS IMPLEMENTATION
// =======================
function [labels, centers] = kmeans(data, k, max_iter)
    [lhs, rhs] = argn();
    if rhs < 3 then
        max_iter = 10;
    end

    [n_samples, n_features] = size(data);

    rand("seed", getdate("s"));
    indices = grand(1, k, "uin", 1, n_samples);
    centers = data(indices, :);

    labels = zeros(n_samples, 1);

    for iter = 1:max_iter
        for i = 1:n_samples
            dists = zeros(1, k);
            for j = 1:k
                dists(j) = norm(data(i,:) - centers(j,:));
            end
            [min_val, min_idx] = min(dists);
            labels(i) = min_idx;
        end

        for j = 1:k
            cluster_points = data(find(labels==j), :);
            if size(cluster_points,1) > 0 then
                centers(j,:) = mean(cluster_points, "r");
            end
        end
    end
endfunction

// =======================
// GUI CREATION
// =======================
function create_gui()
    global img_handle gray_handle seg_handle thr_handle;

    f = figure("figure_name", "Image Segmentation", ...
               "position", [100, 100, 1000, 600], ...
               "background", color("light gray"), ...
               "auto_resize", "on");

    ctrlFrame = uicontrol(f, ...
      'style', 'frame', ...
      'position', [10, 10, 280, 580], ...
      'background', [0.95, 0.95, 0.95], ...
      'Tag', 'ctrlFrame');

    control_panel(ctrlFrame);

    // Axes
    img_handle = newaxes();
    set(img_handle, "axes_bounds", [0.32 0.01 0.32 0.42]);
    xtitle("Original Image");

    gray_handle = newaxes();
    set(gray_handle, "axes_bounds", [0.68 0.01 0.32 0.42]);
    xtitle("Grayscale Image");

    seg_handle = newaxes();
    set(seg_handle, "axes_bounds", [0.32 0.55 0.32 0.42]);
    xtitle("Segmented Image (K-means)");

    thr_handle = newaxes();
    set(thr_handle, "axes_bounds", [0.69 0.55 0.32 0.42]);
    xtitle("Segmented Image (Thresholding)");
endfunction

// =======================
// CONTROL PANEL
// =======================
function control_panel(parent)
    ph = get(parent, 'position');

    uicontrol(parent, 'style', 'text', ...
        'position', [10, ph(4)-40, ph(3)-20, 30], ...
        'string', 'Simulation Settings', ...
        'fontsize', 14, 'fontweight', 'bold');

    uicontrol(parent, "style", "pushbutton", ...
              "string", "Upload Image", ...
              "position", [10, ph(4)-90, 120, 30], ...
              'fontsize', 14, ...
              "callback", "upload_image();");

    uicontrol(parent, 'style','text', ...
        'position',[10, ph(4)-140, 180, 20], ...
        'string','Image Processing Method:',...
         'fontsize', 14,'horizontalalignment','left');           

    uicontrol(parent, "style", "pushbutton", ...
              "string", "Grayscale", ...
              "position", [10, ph(4)-190, 120, 30], ...
              'fontsize', 14, ...
              "callback", "apply_grayscale();");

    uicontrol(parent, 'style','text', ...
        'position',[40, ph(4)-230, 180, 20], ...
        'string','Image Segmentation',...
         'fontsize', 14,'horizontalalignment','center');  
// K-means
    uicontrol(parent, 'style', 'text', ...
        'position', [10, ph(4)-260, 100, 20], ...
        'string', 'K-Means', 'fontsize', 14, ...
        'horizontalalignment', 'left');

    uicontrol(parent, "style", "text", ...
              "string", "k-Size:", ...
              'position', [10, ph(4)-290, 50, 20]);

    uicontrol(parent, "style", "edit", ...
              'position', [60, ph(4)-290, 50, 20], ...
              "tag", "k_input");
    
    uicontrol(parent, 'style', 'pushbutton', ...
        'position', [150, ph(4)-290, 120, 30], ...
        'string', 'Run K-means', ...
        'callback', 'run_kmeans()');          

// Thersholding

    uicontrol(parent, 'style','text', ...
        'position',[10, ph(4)-350, 140, 20], ...
        'string','Thresholding',...
         'fontsize', 14,'horizontalalignment','left'); 
          
    uicontrol(parent, 'style', 'pushbutton', ...
        'position', [150, ph(4)-350, 120, 30], ...
        'string', 'Run Thershold', ...
        'callback', ' run_thresholding()');                 

   // Actions
    uicontrol(parent, 'style', 'text', ...
        'position', [10, ph(4)-400, ph(3)-20, 20], ...
        'string', 'Actions', 'fontsize', 12, ...
        'fontweight', 'bold', 'horizontalalignment', 'left');

    uicontrol(parent, 'style', 'pushbutton', ...
        'position', [10, ph(4)-450, 120, 30], ...
        'string', 'Clear Segment', ...
        'callback', 'clear_segment_callback()');

    uicontrol(parent, 'style', 'pushbutton', ...
        'position', [150, ph(4)-450, 120, 30], ...
        'string', 'Clear all', ...
        'callback', 'clear_axes_callback()');

    uicontrol(parent, 'style', 'pushbutton', ...
        'position', [60, ph(4)-500, 120, 30], ...
        'string', 'Exit', ...
        'callback', 'exit_callback()');
endfunction

// =======================
// CALLBACKS
// =======================
function upload_image()
    global img_handle original_img;
    [filename, path] = uigetfile("*.jpg;*.png;*.bmp", "Select an image");
    if filename <> "" then
        img_path = path + "\" + filename;
        original_img = imread(img_path);
        delete(get(img_handle, "children"));
        sca(img_handle);
        imshow(original_img);
    end
endfunction

function apply_grayscale()
    global original_img gray_handle;
    if exists("original_img", "local") then
        gray_img = rgb2gray(original_img);
        delete(get(gray_handle, "children"));
        sca(gray_handle);
        imshow(gray_img);
    else
        messagebox("Please upload an image first.", "Warning", "warning");
    end
endfunction

function run_thresholding()
         global original_img seg_handle;

    if ~exists("original_img", "local") then
        messagebox("Please upload an image first.", "Warning", "warning");
        return;
    end

    gray = rgb2gray(original_img);
    gray_d = double(gray);

    T = mean(gray_d);

    binary_img = gray_d > T;
    binary_img = uint8(binary_img) * 255;

    delete(get(thr_handle, "children"));
    sca(thr_handle);
    imshow(binary_img);
endfunction

function run_kmeans()
    global original_img seg_handle;

    if ~exists("original_img", "local") then
        messagebox("Please upload an image first.", "Warning", "warning");
        return;
    end

    // Run K-means
    k_input = findobj("tag", "k_input");
    k = evstr(get(k_input, "string"));

    if or(k <= 0 | k <> round(k)) then
        messagebox("Please enter a valid positive integer for k.", "Error", "error");
        return;
    end

    img = imresize(original_img, [100 100]);
    [rows, cols, ch] = size(img);
    pixels = matrix(img, rows*cols, ch);
    pixels = double(pixels);

    [labels, centers] = kmeans(pixels, k);

    segmented = centers(labels, :);
    segmented_img = matrix(uint8(segmented), rows, cols, ch);

    delete(get(seg_handle, "children"));
    sca(seg_handle);
    imshow(segmented_img);
endfunction

function clear_axes_callback()
    global img_handle gray_handle seg_handle thr_handle original_img;
    original_img = [];
    delete(get(img_handle, "children"));
    delete(get(gray_handle, "children"));
    delete(get(seg_handle, "children"));
    delete(get(thr_handle, "children"));
endfunction

function clear_segment_callback()
    global img_handle gray_handle seg_handle thr_handle original_img;     
    delete(get(seg_handle, "children"));
    delete(get(thr_handle, "children"));
endfunction

function exit_callback()
    close(gcf());
endfunction


// =======================
// LAUNCH
// =======================
create_gui();
