// Start timer
tic();

// Clear previous figures and variables
clear;
clf;

// Default values
default_mach = 2;
default_gamma = 1.4;
default_steps = 50;
default_guess = 0.375*%pi/180;
default_plot_text = %f;

// Ask user for inputs with default options
printf("\nNozzle Contour Calculator\n");
printf("Press Enter to use default values, or input new values:\n\n");

// Mach number input
user_input = input("Enter Mach exit number (default = " + string(default_mach) + "): ", "string");
if user_input == "" then
    mach_exit = default_mach;
else
    mach_exit = evstr(user_input);
end

// Gamma input
user_input = input("Enter specific heat ratio gamma (default = " + string(default_gamma) + "): ", "string");
if user_input == "" then
    gamma = default_gamma;
else
    gamma = evstr(user_input);
end

// Steps input
user_input = input("Enter number of steps (default = " + string(default_steps) + "): ", "string");
if user_input == "" then
    steps = default_steps;
else
    steps = evstr(user_input);
end

// Use default values for guess and plot_text
guess = default_guess;
plot_text = default_plot_text;

// Display final values
printf("\nUsing the following values:\n");
printf("Mach exit = %.2f\n", mach_exit);
printf("Gamma = %.2f\n", gamma);
printf("Steps = %d\n", steps);
printf("\nCalculating nozzle contour...\n\n");

// Prandtl-Meyer function
function nu = prandtl_meyer(mach, gamma)
    nu = sqrt((gamma+1)/(gamma-1))*atan(sqrt((gamma-1)/(gamma+1)*(mach^2-1)))-atan(sqrt(mach^2-1));
endfunction

//% OUTPUT VALUES
theta_wmax = prandtl_meyer(mach_exit,gamma)/2;
delta_theta = (theta_wmax-guess)/steps;

if plot_text then
    col = 'yellow';
else
    col = 'black';
end

// Number of points
nop = 0;
for i = 1:steps+1
    nop = nop + steps + 2 - (i-1);
end

data = zeros(nop,6); // point properties
XY = zeros(nop,2);   // XY location - non dimensional throat radius=1

// Series function
function s = sel(x)
    s = steps + 3 + x*steps - (x+1)*x/2 + 2*x;
endfunction

//% Point properties and location
// FIRST ROUND: compute K- and K+ (characteristic constants)
function y = pm_solve(x, theta, gamma)
    y = prandtl_meyer(x,gamma) - theta;
endfunction

for i = 1:steps+1
    theta = guess + (i-1)*delta_theta;
    data(i,3) = theta;
    // Use fsolve with modified interface
    data(i,5) = fsolve(1.1, list(pm_solve, theta, gamma));
    data(i,6) = asin(1/data(i,5));
    data(i,4) = prandtl_meyer(data(i,5),gamma);
    data(i,1) = data(i,3) + data(i,4);
    data(i,2) = data(i,3) - data(i,4);
end

// FIRST ROUND compute points on the first characteristic line
data(steps+2,:) = data(steps+1,:); // SIMPLE REGION
XY(1,1) = -1/tan(data(1,3)-data(1,6));

// Create new figure window
f = scf();
clf(f); // Clear the figure

// Initial plot
plot([0 XY(1,1)], [1 XY(1,2)], 'b-');
xgrid;

for i = 2:steps+1
    theta12 = 0.5*(data(i-1,3)+data(i,3));
    mu12 = 0.5*(data(i-1,6)+data(i,6));
    mu2 = data(i,6);
    theta2 = data(i,3);
    XY(i,1) = (XY(i-1,2)-1-tan(theta12+mu12)*XY(i-1,1))/(tan(theta2-mu2)-tan(theta12+mu12));
    XY(i,2) = XY(i-1,2)+tan(theta12+mu12)*(XY(i,1)-XY(i-1,1));
    plot([XY(i-1,1) XY(i,1)], [XY(i-1,2) XY(i,2)], 'b-');
    plot([0 XY(i,1)], [1 XY(i,2)], 'b-');
end

// First contour point
i = steps+2;
theta12 = 0.5*(data(i-1,3)+data(i,3));
mu12 = 0.5*(data(i-1,6)+data(i,6));
thetaai = 0.5*(theta_wmax+data(i,3));
XY(i,1) = (XY(i-1,2)-1-tan(theta12+mu12)*XY(i-1,1))/(tan(thetaai)-tan(theta12+mu12));
XY(i,2) = XY(i-1,2)+tan(theta12+mu12)*(XY(i,1)-XY(i-1,1));
plot([XY(i-1,1) XY(i,1)], [XY(i-1,2) XY(i,2)], 'b-');
plot([0 XY(i,1)], [1 XY(i,2)], 'k-', 'LineWidth', 3);

// SECOND ROUND: compute the non-simple region
for j = 0:steps-1
    // CENTERLINE
    theta = 0;
    data(sel(j),1) = data(j+2,1); // K-
    data(sel(j),4) = data(sel(j),1)+data(sel(j),3); // nu
    data(sel(j),2) = theta-data(sel(j),4); // K+
    data(sel(j),5) = fsolve(1.1, list(pm_solve, data(sel(j),4), gamma)); // mach2
    data(sel(j),6) = asin(1/data(sel(j),5)); // mu
    
    // NON-SIMPLE REGION - moving along the K+ characteristic
    for i = 1:steps-j-1
        data(sel(j)+i,3) = i*delta_theta; // no guess
        data(sel(j)+i,1) = data(j+i+2,1); // K-
        data(sel(j)+i,2) = data(sel(j),2); // K+
        data(sel(j)+i,4) = 0.5*(data(sel(j)+i,1)-data(sel(j)+i,2)); // nu
        data(sel(j)+i,5) = fsolve(1.1, list(pm_solve, data(sel(j)+i,4), gamma)); // mach2
        data(sel(j)+i,6) = asin(1/data(sel(j)+i,5)); // mu
    end
    data(sel(j)+steps-j,:) = data(sel(j)+steps-j-1,:); // SIMPLE REGION
    
    // DRAW CHARACTERISTIC LINES
    // Draw centerline
    theta13 = 0.5*(data(sel(j-1)+1,3)+data(sel(j),3));
    mu13 = 0.5*(data(sel(j-1)+1,6)+data(sel(j),6));
    XY(sel(j),1) = XY(sel(j-1)+1,1)-XY(sel(j-1)+1,2)/tan(theta13-mu13);
    plot([XY(sel(j-1)+1,1) XY(sel(j),1)], [XY(sel(j-1)+1,2) XY(sel(j),2)], 'b-');
    
    for i = 1:steps-j-1
        theta12 = 0.5*(data(sel(j)+i,3)+data(sel(j)+i-1,3));
        mu12 = 0.5*(data(sel(j)+i,6)+data(sel(j-1)+i-1,6));
        theta23 = 0.5*(data(sel(j)+i,3)+data(sel(j-1)+i+1,3));
        mu23 = 0.5*(data(sel(j)+i,6)+data(sel(j-1)+i+1,6));
        XY(sel(j)+i,1) = (XY(sel(j)+i-1,2)-XY(sel(j-1)+i+1,2)-tan(theta12+mu12)*XY(sel(j)+i-1,1)...
            +tan(theta23-mu23)*XY(sel(j-1)+i+1,1))/(tan(theta23-mu23)-tan(theta12+mu12));
        XY(sel(j)+i,2) = XY(sel(j)+i-1,2)+tan(theta12+mu12)*(XY(sel(j)+i,1)-XY(sel(j)+i-1,1));
        plot([XY(sel(j-1)+1+i,1) XY(sel(j)+i,1)], [XY(sel(j-1)+i+1,2) XY(sel(j)+i,2)], 'b-');
        plot([XY(sel(j)+i-1,1) XY(sel(j)+i,1)], [XY(sel(j)+i-1,2) XY(sel(j)+i,2)], 'b-');
    end
    
    // DRAW CHARACTERISTIC LINES - SIMPLE REGION - CONTOUR
    theta12 = 0.5*(data(sel(j+1)-1,3)+data(sel(j+1)-2,3));
    mu12 = 0.5*(data(sel(j+1)-1,6)+data(sel(j+1)-2,6));
    theta23 = 0.5*(data(sel(j)-1,3)+data(sel(j+1)-1,3));
    XY(sel(j+1)-1,1) = (XY(sel(j+1)-2,2)-XY(sel(j)-1,2)-tan(theta12+mu12)*XY(sel(j+1)-2,1)...
        +tan(theta23)*XY(sel(j)-1,1))/(tan(theta23)-tan(theta12+mu12));
    XY(sel(j+1)-1,2) = XY(sel(j+1)-2,2)+tan(theta12+mu12)*(XY(sel(j+1)-1,1)-XY(sel(j+1)-2,1));
    plot([XY(sel(j+1)-2,1) XY(sel(j+1)-1,1)], [XY(sel(j+1)-2,2) XY(sel(j+1)-1,2)], 'b-');
    plot([XY(sel(j)-1,1) XY(sel(j+1)-1,1)], [XY(sel(j)-1,2) XY(sel(j+1)-1,2)], 'k-', 'LineWidth', 3);
end

// Set aspect ratio and labels
set(gca(), "isoview", "on");
title("Minimum-length nozzle contour", "fontsize", 4);
xlabel("x/r_{throat}", "fontsize", 3);
ylabel("y/r_{throat}", "fontsize", 3);

// EXPORT
export = zeros(steps+2,3);
export(:,1) = 1; // Set first column to 1
export(1,:) = [1 0 1]; // First row as initial condition

// Fill in export matrix with XY coordinates
for j = -1:steps-1
    export(j+3,2) = XY(sel(j+1)-1,1); // X position
    export(j+3,3) = XY(sel(j+1)-1,2); // Y position
end

// File export with error handling
try
    // File path for saving the CSV
    file_path = 'E:\Matlab\Scilab\nozzle.csv';
    
    // Create a string matrix for formatted output
    [rows, cols] = size(export);
    export_str = matrix([], rows, cols);
    
    // Format each number with 6 decimal places
    for i = 1:rows
        for j = 1:cols
            export_str(i,j) = sprintf('%.6f', export(i,j));
        end
    end
    
    // Write to CSV file
    [fd, err] = mopen(file_path, 'wt');
    if err ~= 0 then
        error('Could not open file for writing. Check path and permissions.');
    end
    
    for i = 1:rows
        line = strcat(export_str(i,:), ';');
        mfprintf(fd, '%s\n', line);
    end
    mclose(fd);
    
    printf('\nResults successfully exported to: %s\n', file_path);
catch
    printf('\nError: Failed to save results. Check file path and permissions.\n');
    printf('Attempted to save to: %s\n', file_path);
end

// Get execution time and print final messages
execution_time = toc();
printf('\nExecution time: %.3f seconds\n', execution_time);
printf('Nozzle contour calculation completed successfully!\n');

