//------------------------------------------------------------------
// Initialize Fields
//------------------------------------------------------------------

// u: (Nx+1) x (Ny+2) because u-velocity is stored at faces in y-direction.
u = zeros(Nx+1, Ny+2);
v = zeros(Nx+2, Ny+1);
uStar  = zeros(Nx+1, Ny+2);
vStar  = zeros(Nx+2, Ny+1);
uPrime = zeros(Nx+1, Ny+2);
vPrime = zeros(Nx+2, Ny+1);
dU     = zeros(Nx+1, Ny+2);
dV     = zeros(Nx+2, Ny+1);

// Temperature and Pressure fields: (Nx+2) x (Ny+2)
T      = zeros(Nx+2, Ny+2);
p      = zeros(Nx+2, Ny+2);
pStar  = zeros(Nx+2, Ny+2);
pPrime = zeros(Nx+2, Ny+2);

// Global Coefficient Matrices (size based on internal nodes)
Fw = zeros(Nx+1, Ny+1);
Fe = zeros(Nx+1, Ny+1);
Fn = zeros(Nx+1, Ny+1);
Fs = zeros(Nx+1, Ny+1);
DF = zeros(Nx+1, Ny+1);
aW = zeros(Nx+1, Ny+1);
aE = zeros(Nx+1, Ny+1);
aS = zeros(Nx+1, Ny+1);
aN = zeros(Nx+1, Ny+1);
aP = zeros(Nx+1, Ny+1);
bP = zeros(Nx+1, Ny+1);

// Residual Histories
ures = zeros(NmaxSIM, 1);
vres = zeros(NmaxSIM, 1);
pres = zeros(NmaxSIM, 1);

//------------------------------------------------------------------
// Set Inlet Velocity (Uniform Inlet)
u(:, Ju) = U;

// Initialize Pressure Field with a Linear Drop (Fully Developed Flow)
p1 = 12 * mu * U * L / ((2 * H)^2);
linP = linspace(p1, 0, Nx);
for i = 1:Nx
    for j = 1:Ny
        // p is stored on a (Nx+2) x (Ny+2) grid; interior nodes offset by 1.
        p(i+1, j+1) = linP(i);
    end
end

// Initialize Temperature Field: Inlet and Wall Temperatures
T(:, Jp)   = Ti;
T(:, 1)    = Tw;
T(:, Ny+2) = Tw;

// Diffusion Coefficients for Momentum Equations (constant)
Dx = (mu/dx) * dy * dz;
Dy = (mu/dy) * dx * dz;

//------------------------------------------------------------------
// Prompt for Temperature Equation Solution
//------------------------------------------------------------------
solve_temp = input("Do you want to solve the temperature equation? (Y/N): ", "string");


//------------------------------------------------------------------
// SIMPLE Algorithm Iterations
//------------------------------------------------------------------
disp("Starting SIMPLE iterations...");
for n = 1:NmaxSIM
    // Store old values
    uOld = u;
    vOld = v;
    pStar = p;
    
    // --- Step 1a: Solve x-momentum for uStar ---
    u = FVM_u(Nx, Ny, dx, dy, dz, rho, Dx, Dy, iF, Ju, alphaU, u, v, pStar, BC_S);
    [uStar, ures(n)] = FVM_GS_ext_mesh(Nx, Ny+1, alphaU, NmaxGSI, tol, u);
    
    // --- Step 1b: Solve y-momentum for vStar ---
    v = FVM_v(Nx, Ny, dx, dy, dz, rho, Dx, Dy, Iv, jF, alphaU, u, v, pStar);
    [vStar, vres(n)] = FVM_GS_ext_mesh(Nx+1, Ny, alphaU, NmaxGSI, tol, v);
    
    // --- Step 2: Pressure Correction Equation ---
    FVM_pcorr(Nx, Ny, dx, dy, dz, rho, Ip, Jp, uStar, vStar);
    pPrime = zeros(Nx+2, Ny+2);
    [pPrime, pres(n)] = FVM_GS_ext_mesh(Nx+1, Ny+1, 1.0, NmaxGSI, tol, pPrime);
    
    // --- Step 3: Correct Pressure and Velocity Fields ---
    p(Ip, Jp) = pStar(Ip, Jp) + alphaP * pPrime(Ip, Jp);
    u(iF, Ju) = uStar(iF, Ju) + dU(iF, Ju).*(pPrime(iF, Ju) - pPrime(iF+1, Ju));
    v(Iv, jF) = vStar(Iv, jF) + dV(Iv, jF).*(pPrime(Iv, jF) - pPrime(Iv, jF+1));
    
// --- Step 4: Check Convergence ---
    if n > 0 then
        // Add this line to plot residuals
        residuals_plot(n, ures, vres, pres);
        
        disp("Iteration " + string(n) + " residuals: u=" + string(ures(n)) + " v=" + string(vres(n)) + " p=" + string(pres(n)));
        cTest = max([ures(n), vres(n)]);
        if cTest < tol then
            disp("Convergence achieved.");
            break;
        elseif cTest > div | isnan(cTest) then
            disp("Residuals are too high. Divergence detected.");
            break;
        end
    end
    
    // Apply Outlet Boundary Conditions (du/dx = 0, dv/dx = 0)
    u(Nx+1, :) = u(Nx, :);
    v(Nx+2, :) = v(Nx+1, :);

    // --- Step 5: Solve Temperature Equation (Inside SIMPLE Iterations) ---
    if solve_temp == "Y" | solve_temp == "y" then
        FVM_phi(Nx, Ny, dx, dy, dz, rho, kt/cp, qw/cp, Ip, Jp, u, v, BC_S, BC_N);
        [T, Tres] = FVM_GS_ext_mesh(Nx+1, Ny+1, 1.0, NmaxGSI, tol, T);
    end
end

// Final Message
if solve_temp == "Y" | solve_temp == "y" then
    disp("Temperature equation solved.");
else
    disp("Skipping temperature equation solution.");
end
