// ========== MONTE CARLO SIMULATION ==========
function estimatePi(N)
    insideCircle = 0;
    xInside = []; yInside = [];
    xOutside = []; yOutside = [];
    
    for i = 1:N
        x = rand();
        y = rand();
        if (x^2 + y^2 <= 1)
            insideCircle = insideCircle + 1;
            xInside = [xInside, x];
            yInside = [yInside, y];
        else
            xOutside = [xOutside, x];
            yOutside = [yOutside, y];
        end
    end
    
    estimatedPi = 4 * (insideCircle / N);
    disp("Estimated Pi: " + string(estimatedPi));
    
    // Plotting
    scf();
    plot(xInside, yInside, "g.");
    plot(xOutside, yOutside, "r.");
    xlabel("X");
    ylabel("Y");
    xtitle("Monte Carlo Estimation of Pi (N="+string(N)+")", "X", "Y");
    a = gca();
    a.data_bounds = [0, 0; 1, 1];
    legend(["Inside Circle"; "Outside Circle"], 2);
endfunction

// Execute with 1,000 samples
estimatePi(1000);



function samples = sobolQMC(N, d)
    global sobolWarned;

    if ~isdef("sobolWarned") then
        sobolWarned = %f;
    end

    if ~isdef("lowdisc_sobol") then
        if ~sobolWarned then
            disp("⚠️ Sobol sequence requires the ''lowdisc'' module. Skipping Sobol.");
            sobolWarned = %t;
        end
        samples = [];
        return;
    end

    s = lowdisc_soboluggest(d); // Initialize Sobol generator
    samples = lowdisc_sobol(N + 1, s.dim, s.skip, s.leap);
    samples = samples(2:$, :); // remove the first (0,...)
endfunction

function plotSampling(samples, titleStr, filename)
    if isempty(samples) then
        disp("⚠️ No samples to display.");
        return;
    end

    clf(); // clear figure
    plot(samples(:,1), samples(:,2), 'ro');
    title(titleStr);
    xtitle("Dimension 1", "Dimension 2");
    xgrid();

    if ~isempty(filename) then
        xs2png(gcf(), filename);
        disp("✅ Saved: " + filename);
    end
endfunction
