clc;
clear;

// Parameters
P_T = 10;               // Transmit power (dBm)
f_c = 2.4e9;            // Carrier frequency (Hz) (e.g., 2.4 GHz)
h = 50;                // UAV height (m)
alpha_LoS = 1;          // Path loss exponent for LoS
alpha_NLoS = 20;        // Path loss exponent for NLoS
sigma = 4;              // Shadowing standard deviation (dB)
a_env = 9.61;           // LoS parameter (urban/suburban/rural)
b_env = 0.16;           // LoS parameter (urban/suburban/rural)
P_threshold = -90;     // Minimum acceptable received power (dBm)
c = 3e8;                // Speed of light (m/s)

// Free space path loss at reference distance
PL_0 = 20 * log10((4 * %pi * f_c) / c);

// Function for LoS Probability
function P_LoS = LoS_Probability(h, r, a_env, b_env)
    theta = atan(h / r) * (180 / %pi);  // Elevation angle in degrees
    P_LoS = 1 / (1 + a_env * exp(-b_env * (theta - a_env)));
endfunction

// Function for Path Loss (LoS and NLoS)
function PL = Path_Loss(d, PL_0, alpha, sigma)
    shadowing = grand(1, 1, "nor", 0, sigma);  // Random shadowing
    PL = PL_0 + 20 * log10(d) + alpha+shadowing;
endfunction

// Function to calculate coverage radius
function r_max = Calculate_Coverage_Radius(P_T, P_threshold, PL_0, alpha_LoS, alpha_NLoS, h, a_env, b_env)
    r = 1;  // Initial radius (m)
    step = 1;  // Increment step (m)
    while %t  // Loop until condition is met
        d = sqrt(h^2 + r^2);  // Distance between UAV and receiver
        P_LoS = LoS_Probability(h, r, a_env, b_env);
        PL_LoS = Path_Loss(d, PL_0, alpha_LoS, 0);
        PL_NLoS = Path_Loss(d, PL_0, alpha_NLoS, 0);
        P_L = P_LoS * PL_LoS + (1 - P_LoS) * PL_NLoS;  // Combined path loss        
        P_R = P_T - P_L;  // Received power        
        if P_R < P_threshold then
            break;
        end
        r = r + step;  // Increase radius
    end
    r_max = r - step;  // Maximum radius before threshold is violated
endfunction

// Calculate coverage radius
r_max = Calculate_Coverage_Radius(P_T, P_threshold, PL_0, alpha_LoS, alpha_NLoS, h, a_env, b_env);

// Calculate coverage area
A_coverage = %pi * r_max^2;

// Display results
disp("Maximum Coverage Radius (m): " + string(r_max));
disp("Coverage Area (m^2): " + string(A_coverage));

// Simulation Parameters
num_users = 50;          // Number of ground users
area_size = 2 * r_max;    // Size of the simulation area (meters)

// Generate Random User Locations
x_users = area_size * rand(1, num_users) - r_max;
y_users = area_size * rand(1, num_users) - r_max;
uav_x = 0; // UAV x-coordinate (center)
uav_y = 0; // UAV y-coordinate (center)

// Check Coverage
user_distances = sqrt(x_users.^2 + y_users.^2);
is_covered = user_distances <= r_max;

// Plot UAV Coverage
clf;
f = gcf(); // Get the current figure
f.color_map = jet(256); // Set the color map

// Draw the coverage circle
theta = linspace(0, 2 * %pi, 100);
x_circle = r_max * cos(theta);
y_circle = r_max * sin(theta);
xgrid
plot(x_circle, y_circle, "cyan", "LineWidth", 2); // Coverage area (circle)

// Plot all users
plot(x_users, y_users, "k.", "MarkerSize", 10); // Uncoverd users 

// Highlight covered users
covered_users = find(is_covered);
plot(x_users(covered_users), y_users(covered_users), "r.", "MarkerSize", 10); // Covered users

// Plot UAV position
plot(uav_x, uav_y, "bd", "MarkerSize", 14); // Red asterisk for UAV position

// Add title and labels
xtitle("UAV Coverage Simulation", "X Coordinate (meters)", "Y Coordinate (meters)");
legend(["UAV Coverage", "Uncovered Users", "Covered Users", "UAV location"]);

