// MIMO, MISO, SIMO and SISO Capacity Comparison GUI with Variable Antennas
clc;
clear all;
funcprot(0);

// Global variable to store legend entries
global legend_entries;
legend_entries = [];

//======================================================================
// Main GUI: left control pane and right graph pane
//======================================================================
function create_gui()
    // Main window (1000×600)
    f = figure(...
      "figure_name",  "MIMO vs SISO Performance Comparison", ...
      "position",     [100, 100, 1000, 600], ...
      "background",   color("light gray"),      ...
      "auto_resize",  "on"                      ...
    );

    // Left controls frame
    ctrlFrame = uicontrol(f, ...
      'style',      'frame', ...
      'position',   [10, 10, 280, 580], ...
      'background', [0.95, 0.95, 0.95], ...
      'Tag',        'ctrlFrame' ...
    );
    create_control_panel(ctrlFrame);

    // Right graph frame
    plotFrame = uicontrol(f, ...
      'style',      'frame', ...
      'position',   [320, 10, 670, 580], ...
      'background', [1, 1, 1], ...
      'Tag',        'plotFrame' ...
    );
    // Create axes inside plotFrame
    ax = newaxes(plotFrame);
    ax.Tag = "plotAxes";
    ax.margins = [0.1 0.1 0.1 0.1];
    ax.axes_bounds = [0 0 1 1];
    ax.grid = [1 1];
endfunction

//======================================================================
// Control panel contents 
//======================================================================
function create_control_panel(parent)
    ph = get(parent, 'position'); // [x y w h]

    // Title
    uicontrol(parent, 'style', 'text', ...
        'position', [10, ph(4)-40, ph(3)-20, 30], ...
        'string', 'Simulation Parameters', ...
        'fontsize', 14, ...
        'fontweight', 'bold');
        
    // Channel Model
    uicontrol(parent, 'style','text', ...
        'position',[10, ph(4)-80, 120, 20], ...
        'string','Channel Model:',...
        'fontsize', 14,'horizontalalignment','left');
    uicontrol(parent, 'style','radiobutton', ...
      'string','AWGN','position',[10, ph(4)-120, 100, 20], ...
      'value',1,'callback','channelCallback()','Tag','awgnBtn');
    uicontrol(parent, 'style','radiobutton', ...
      'string','Rayleigh','position',[120, ph(4)-120, 100, 20], ...
      'callback','channelCallback()','Tag','rayBtn');
      
   // System Configuration Dropdown
    uicontrol(parent, 'style','text', ...
        'position',[10, ph(4)-160, 140, 20], ...
        'string','System Configuration:',...
         'fontsize', 14,'horizontalalignment','left');   

    // Tx Antennas
    uicontrol(parent, 'style','text', ...
        'position',[10, ph(4)-200, 120, 20], ...
        'string','Tx Antennas:',...
        'fontsize', 14,'horizontalalignment','left');
    uicontrol(parent, 'style','popupmenu', ...
        'position',[130, ph(4)-205, 100, 25], ...
        'string',['1';'2';'3';'4'], ...
        'value',1, 'Tag','txAnt');

    // Rx Antennas
    uicontrol(parent, 'style','text', ...
        'position',[10, ph(4)-240, 120, 20], ...
        'string','Rx Antennas:',...
        'fontsize', 14,'horizontalalignment','left');
    uicontrol(parent, 'style','popupmenu', ...
        'position',[130, ph(4)-245, 100, 25], ...
        'string',['1';'2';'3';'4'], ...
        'value',1, 'Tag','rxAnt');

    // SNR Range
    uicontrol(parent, 'style','text', ...
        'position',[10, ph(4)-290,120,20], ...
        'string','SNR Range (dB):',...
        'fontsize', 14,'horizontalalignment','left');
    uicontrol(parent, 'style','edit', ...
        'position',[10, ph(4)-340,50,25], ...
        'string','0','Tag','snrMin');
    uicontrol(parent,'style','text', ...
        'position',[70, ph(4)-340,20,25], ...
        'string','to','horizontalalignment','center');
    uicontrol(parent,'style','edit', ...
        'position',[90, ph(4)-340,50,25], ...
        'string','20','Tag','snrMax');

    // Action Buttons Panel Title
    uicontrol(parent, 'style', 'text', ...
        'position', [10, ph(4)-380, ph(3)-20, 20], ...
        'string', 'Actions', ...
        'fontsize', 12, ...
        'fontweight', 'bold', ...
        'horizontalalignment','left');

    // Run Simulation Button
    uicontrol(parent, 'style','pushbutton', ...
        'position',[10, ph(4)-420,120,30], ...
        'string','Run Simulation','callback','run_simulation_callback()');

    // Clear Figure Button
    uicontrol(parent, 'style','pushbutton', ...
        'position',[150, ph(4)-420,120,30], ...
        'string','Clear Figure','callback','clear_axes_callback()');
        
    // Exit Button
    uicontrol(parent, 'style','pushbutton', ...
        'position',[60, ph(4)-480,120,30], ...
        'string','Exit','callback','exit_callback()');
endfunction

//======================================================================
// Channel radio button callback
//======================================================================
function channelCallback()
    btns = findobj("Tag","awgnBtn",'-or',"Tag","rayBtn");
    clicked = gcbo();
    for b = btns'
        if b == clicked then b.value = 1; else b.value = 0; end
    end
endfunction

//======================================================================
// Simulation model (variable Nt and Nr)
//======================================================================
function [data_rate] = simulate_system(channel, snr_db, Nt, Nr)
    snr = 10^(snr_db/10);
    if channel == "AWGN" then
        // AWGN capacity
        if Nt == 1 & Nr == 1 then
            data_rate = log2(1 + snr);
        elseif  Nt == 2 & Nr == 1
            data_rate = log2(1 + snr*1.2);
        elseif Nt == 1 & Nr == 2    
             data_rate = log2(1 + snr*Nr);       
        elseif Nt >= 2 & Nr >= 2    
             data_rate = min(Nt, Nr) * log2(1 + snr);
        end
    else
        // Rayleigh fading Monte Carlo
        N = 1e4; inst = zeros(1, N);
        for k = 1:N
            H = (rand(Nr, Nt, 'normal') + %i*rand(Nr, Nt, 'normal'))/sqrt(2);
            inst(k) = log2(det( eye(Nr) + (snr/Nt) * H * H' ));
        end
        data_rate = mean(inst);
    end
endfunction

//======================================================================
// Run simulation and plot
//======================================================================
function run_simulation_callback()
    global legend_entries;
    fig = gcf();
    // Determine channel
    if get(findobj(fig,'Tag','awgnBtn'),'value') == 1 then channel = "AWGN"; else channel = "Rayleigh"; end
    // Get antenna counts
    txIdx = get(findobj(fig,'Tag','txAnt'),'value');
    rxIdx = get(findobj(fig,'Tag','rxAnt'),'value');
    Nt = txIdx;
    Nr = rxIdx;
    // SNR range
    snr_start = evstr(get(findobj(fig,'Tag','snrMin'),'string'));
    snr_end   = evstr(get(findobj(fig,'Tag','snrMax'),'string'));
    snr_vec   = snr_start:2:snr_end;

    rates = zeros(1, length(snr_vec));
    for i = 1:length(snr_vec)
        rates(i) = simulate_system(channel, snr_vec(i), Nt, Nr);
    end

    ax = findobj('Tag','plotAxes');
    // Define styles
    colors = ['b', 'g', 'r', 'c', 'm', 'y', 'k'];
    line_styles = ['-', '--', ':', '-.'];
    idx = modulo(size(legend_entries,1), size(colors,2)) + 1;
    ls  = modulo(size(legend_entries,1), size(line_styles,2)) + 1;
    line_spec = colors(idx) + line_styles(ls);

    plot(ax, snr_vec, rates, line_spec,'linewidth',2,'marker','o');
    xlabel(ax,'SNR (dB)', 'fontsize',4,'fontweight','bold');
    ylabel(ax,'Data Rate (bits/s/Hz)', 'fontsize',4,'fontweight','bold');
    title(ax,'Capacity Comparison', 'fontsize',4,'fontweight','bold');

    // Update legend
    legend_entries = [legend_entries; string(Nt)+"x"+string(Nr)+" - "+channel];
    legend(ax, legend_entries);
endfunction

//======================================================================
// Clear axes callback
//======================================================================
function clear_axes_callback()
    global legend_entries;
    ax = findobj('Tag','plotAxes');
    children = get(ax,'children');
    if ~isempty(children) then delete(children); end
    legend_entries = [];
    legend(ax,'off');
endfunction

//======================================================================
// Exit callback
//======================================================================
function exit_callback()
    close(gcf());
endfunction

// Launch GUI
create_gui();
